// Author: 		Sally Hudson and Sydnee Caldwell 
// Modified: 	May 26, 2015
// This file creates the tables for "Evaluating Econometric Evaluations of Post-Secondary Aid"

// preamble
clear all
set more off

// data and code directories
local root				// declare local directory path here
local data				`root'\analysis_flat_PNP.dta
sysdir set PERSONAL 	`root'\ado	

// RD parameters
local CC_cutoff = 22
local SC_cutoff = 22
local NU_cutoff = 21
local bw = 4

// bootstrap parameters
local seed = 42
local n_reps = 1000

************************************************************************************
************************************************************************************

*** PREP DATA ***

// load_data
use `data', clear

gen bw = 0
foreach system in NU SC CC {

	// define bandwidths for the RD sample
	replace bw = 1 if s_strata_`system' & (score_total >=``system'_cutoff' - `bw' + 1 & score_total <= ``system'_cutoff' + `bw')
	
	// re-center scores around bandwith
	replace score_total = score_total - ``system'_cutoff' if s_strata_`system'		
}

// Interaction term between running variable and cutoff indicator
gen score_totalXawardoffer = score_total*award_offered

************************************************************************************
************************************************************************************
************************************************************************************

*** TABLE 1: COVARIATE BALANCE ***

// variables to balance
#delimit ;
local vars_CB
	female
	race_white
	EFC
	income_family	
	GPA_HS
;
#delimit cr

// table layout parameters
local n_cols 	= 9
local n_rows 	= 2
local n_stats 	= 2

// sample for columns
#delimit ;
local samples
	s_RCT_STBF_2012
	s_eligible_2011
	"s_cohort_2012 & (s_awardees_MF | (s_RCT & !award_offered)) & bw"
	;
#delimit cr

// initialize top row and sample size matrix
matrix results = J(1, `n_cols', 0)
matrix N = J(1, `n_cols', 0)

// loop over covariates, which vary with rows
foreach cov of local vars_CB {

	// initialize blank row
	matrix row = J(`n_rows', `n_cols', 0)
	
	// loop over samples, which vary with columns
	local col = 0
	foreach s of local samples {

		// treated mean
		sum `cov' if `s' & award_offered
		matrix row[1, 3*`col' + 1] 	= r(mean)
		matrix N[1, 3*`col' + 1] 	= r(N)
		
		// strata-adjusted difference
		regress `cov' award_offered d_strata* if `s', vce(robust)
		matrix row[1, 3*`col' + 3] 	= _b[award_offered]
		matrix row[2, 3*`col' + 3] 	= _se[award_offered]
		matrix N[1, 3*`col' + 3] 	= e(N)
		
		// increment column counter
		local col = `col' + 1
	}
	
	// append row to base matrix
	matrix results = results \ row
}

// append stats and sample size to means
matrix results = results \ N

// convert matrix to data set
preserve
clear
svmat results
matlist results
drop if _n == 1

// export file
export excel tables, sheet(table1_raw) sheetreplace
restore

************************************************************************************
************************************************************************************
************************************************************************************

*** TABLE 2: TREATMENT EFFECTS ***

// Controls
local controls female race_white GPA_HS EFC income_family income_family_miss

// impute family income where missing
gen income_family_miss = missing(income_family)
egen EFC_coarse = cut(EFC), at(0(1000)10000)
bysort EFC_coarse familyMembers*: egen income_mean = mean(income_family)
replace income_family = income_mean if missing(income_family)
bysort EFC_coarse: egen income_mean2 = mean(income_family)
replace income_family = income_mean2 if missing(income_family)
assert !missing(income_family) 

// ensure all other covariates are non-missing
foreach v of varlist `controls' {
	bysort college_considered_primary cohort: egen mu = mean(`v')
	replace `v' = mu if missing(`v')
	drop mu
}

	
// samples
#delimit ;
local samples
	s_RCT_STBF_2012
	"s_eligible_2011"
	"s_RCT_STBF_2012 & bw"
	"s_cohort_2012 & (s_awardees_MF | (s_RCT & !award_offered)) & bw"
	;
#delimit cr

// table layout parameters
local n_ests	= 7
local n_stats 	= 2

// outcome
local y att_4yr_Y2

 
// initialize blank column for table
matrix results = J(2*`n_ests' + `n_stats', 1, 0)
		
// loop over samples, which vary with columns
local i = 1
foreach s of local samples {

	// initialize blank column for sample
	matrix col = J(2*`n_ests' + `n_stats', 1, 0)

	// restrict sample
	preserve 
	keep if `s'
	assert s_all

	// control mean
	sum `y' if !award_offered
	matrix col[1, 1] = r(mean)
	
	// raw difference
	reg `y' award_offered 
	matrix col[2, 1] = _b[award_offered]
	matrix col[3, 1] = _se[award_offered]
	
	// strata matching
	bootstrap ate=r(ate), rep(`n_reps') seed(`seed'): matching `y' award_offered college_considered_primary
	matrix col[4, 1] = _b[ate]
	matrix col[5, 1] = _se[ate]
	
	// strata-adjusted OLS
	reg `y' award_offered d_strata*
	matrix col[6, 1] = _b[award_offered]
	matrix col[7, 1] = _se[award_offered]

	// OLS
	reg `y' award_offered d_strata* `controls'
	matrix col[8, 1] = _b[award_offered]
	matrix col[9, 1] = _se[award_offered]
	
	// RV controls and interaction
	if regexm("`s'", "s_awardees_MF") {
		regress `y' award_offered d_strata* `controls' score_total score_totalXawardoffer
		matrix col[10, 1] = _b[award_offered]
		matrix col[11, 1] = _se[award_offered]
	}
	else {
	
		// HIR
		logit award_offered d_strata* `controls'
		predict pscore, pr
		bootstrap ate=e(ate), rep(`n_reps') seed(`seed'): hir `y' award_offered pscore
		matrix col[12, 1] = _b[ate]
		matrix col[13, 1] = _se[ate]
		
		// Oaxaca-Blinder
		kline `y' award_offered d_strata* `controls'
		matrix col[14, 1] = e(ate)
		matrix col[15, 1] = e(se)
	}	

	// sample size
	matrix col[16, 1] = _N
	
	// store results 
	matrix results = results, col
	restore
}

// convert matrix to data set
preserve
clear
matlist results
svmat results
drop results1

// export file
export excel tables, sheet(table2_raw) sheetreplace
	
************************************************************************************
************************************************************************************
************************************************************************************
